/**
 * VanTurtle Fan Controller Library for ESP32
 *
 * This library provides control for VanTurtle Fan controllers via I2C.
 * Compatible with both RJ45 and RJ11 fan models.
 */

#ifndef VANTURTLEFAN_H
#define VANTURTLEFAN_H

#include <stdint.h>
#include <stdbool.h>
#include "driver/i2c.h"

#ifdef __cplusplus
extern "C" {
#endif

// Fan identifiers
#define FAN_ONE 1
#define FAN_TWO 2

// PCF8575 Register addresses
#define REG_INPUT_PORT_0  0x00
#define REG_INPUT_PORT_1  0x01
#define REG_OUTPUT_PORT_0 0x02
#define REG_OUTPUT_PORT_1 0x03
#define REG_CONFIG_PORT_0 0x06
#define REG_CONFIG_PORT_1 0x07

// Button press duration in milliseconds
#define BUTTON_PRESS_DURATION 150

// Error codes
#define VTF_OK                  0
#define VTF_ERR_INVALID_FAN    -1
#define VTF_ERR_I2C_FAIL       -2
#define VTF_ERR_INVALID_PIN    -3

/**
 * VanTurtle Fan controller structure
 */
typedef struct {
    i2c_port_t i2c_port;
    uint8_t i2c_address;
} vanturtlefan_t;

/**
 * Initialize the VanTurtle Fan controller
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param i2c_port I2C port number (I2C_NUM_0 or I2C_NUM_1)
 * @param sda_pin GPIO pin number for SDA
 * @param scl_pin GPIO pin number for SCL
 * @param i2c_address I2C address of the fan controller (default 0x27)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_init(vanturtlefan_t *vtf, i2c_port_t i2c_port, int sda_pin, int scl_pin, uint8_t i2c_address);

/**
 * Deinitialize the VanTurtle Fan controller and free I2C resources
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_deinit(vanturtlefan_t *vtf);

/**
 * Simulate a button press for the specified fan and pin
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @param pin Pin number to press (1-7)
 * @param length Duration to hold the button in milliseconds (0 = default)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_press(vanturtlefan_t *vtf, uint8_t fan, uint8_t pin, uint16_t length);

/**
 * Check if the Auto ("hold to set") LED is on for the specified fan (RJ45 only)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @param state Pointer to store the LED state (true if on, false if off)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_is_autohold(vanturtlefan_t *vtf, uint8_t fan, bool *state);

// ============================================================================
// RJ45 Model Functions
// ============================================================================

/**
 * Toggle Auto mode on/off (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_auto(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Reverse airflow direction (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_reverse(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Increase airflow speed (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_faster(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Decrease airflow speed (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_slower(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Make the fan beep with no configuration change (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_beep(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Toggle fan on/off (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_onoff(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Reset fan to off state (RJ45)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj45_reset(vanturtlefan_t *vtf, uint8_t fan);

// ============================================================================
// RJ11 Model Functions
// ============================================================================

/**
 * Turn on fan or cycle through speed settings (RJ11)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj11_on(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Close the fan lid (RJ11)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj11_close(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Open the fan lid (RJ11)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj11_open(vanturtlefan_t *vtf, uint8_t fan);

/**
 * Turn off the fan (RJ11)
 *
 * @param vtf Pointer to VanTurtle Fan structure
 * @param fan Fan number (FAN_ONE or FAN_TWO)
 * @return VTF_OK on success, error code otherwise
 */
int vanturtlefan_rj11_off(vanturtlefan_t *vtf, uint8_t fan);

#ifdef __cplusplus
}
#endif

#endif // VANTURTLEFAN_H
