/**
 * VanTurtle Fan Controller Library for ESP32
 */

#include "vanturtlefan.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_log.h"
#include <string.h>

static const char *TAG = "VanTurtleFan";

#define I2C_MASTER_TIMEOUT_MS 1000

// Private helper functions
static uint8_t _get_reg(uint8_t fan);
static int _i2c_write_byte(vanturtlefan_t *vtf, uint8_t reg, uint8_t data);
static int _i2c_read_byte(vanturtlefan_t *vtf, uint8_t reg, uint8_t *data);
static int _set_pin(vanturtlefan_t *vtf, uint8_t fan, uint8_t pin, bool state);

/**
 * Get the register address for the specified fan
 */
static uint8_t _get_reg(uint8_t fan) {
    return (fan == FAN_ONE) ? REG_OUTPUT_PORT_0 : REG_OUTPUT_PORT_1;
}

/**
 * Write a byte to an I2C register
 */
static int _i2c_write_byte(vanturtlefan_t *vtf, uint8_t reg, uint8_t data) {
    uint8_t write_buf[2] = {reg, data};

    esp_err_t ret = i2c_master_write_to_device(
        vtf->i2c_port,
        vtf->i2c_address,
        write_buf,
        sizeof(write_buf),
        pdMS_TO_TICKS(I2C_MASTER_TIMEOUT_MS)
    );

    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "I2C write failed: %s", esp_err_to_name(ret));
        return VTF_ERR_I2C_FAIL;
    }

    return VTF_OK;
}

/**
 * Read a byte from an I2C register
 */
static int _i2c_read_byte(vanturtlefan_t *vtf, uint8_t reg, uint8_t *data) {
    esp_err_t ret = i2c_master_write_read_device(
        vtf->i2c_port,
        vtf->i2c_address,
        &reg,
        1,
        data,
        1,
        pdMS_TO_TICKS(I2C_MASTER_TIMEOUT_MS)
    );

    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "I2C read failed: %s", esp_err_to_name(ret));
        return VTF_ERR_I2C_FAIL;
    }

    return VTF_OK;
}

/**
 * Set the state of a specific pin
 */
static int _set_pin(vanturtlefan_t *vtf, uint8_t fan, uint8_t pin, bool state) {
    if (fan != FAN_ONE && fan != FAN_TWO) {
        return VTF_ERR_INVALID_FAN;
    }

    if (pin < 1 || pin > 7) {
        return VTF_ERR_INVALID_PIN;
    }

    uint8_t reg = _get_reg(fan);
    uint8_t output;

    // Read current output state
    int ret = _i2c_read_byte(vtf, reg, &output);
    if (ret != VTF_OK) {
        return ret;
    }

    // Set or clear the requested pin
    if (state) {
        output |= (1 << pin);
    } else {
        output &= ~(1 << pin);
    }

    // Ensure activity LED (bit 1) reflects other bits
    // If any other bit besides bit 1 is set, set bit 1 to 1; otherwise clear it
    uint8_t other_bits_mask = ~(1 << 1) & 0xFF;
    if (output & other_bits_mask) {
        output |= (1 << 1);
    } else {
        output &= ~(1 << 1);
    }

    // Write the updated state
    return _i2c_write_byte(vtf, reg, output);
}

// Public API functions
int vanturtlefan_init(vanturtlefan_t *vtf, i2c_port_t i2c_port, int sda_pin, int scl_pin, uint8_t i2c_address) {
    if (vtf == NULL) {
        return VTF_ERR_I2C_FAIL;
    }

    vtf->i2c_port = i2c_port;
    vtf->i2c_address = i2c_address;

    // Configure I2C
    i2c_config_t conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = sda_pin,
        .scl_io_num = scl_pin,
        .sda_pullup_en = GPIO_PULLUP_ENABLE,
        .scl_pullup_en = GPIO_PULLUP_ENABLE,
        .master.clk_speed = 100000, // 100kHz
    };

    esp_err_t ret = i2c_param_config(i2c_port, &conf);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "I2C param config failed: %s", esp_err_to_name(ret));
        return VTF_ERR_I2C_FAIL;
    }

    ret = i2c_driver_install(i2c_port, conf.mode, 0, 0, 0);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "I2C driver install failed: %s", esp_err_to_name(ret));
        return VTF_ERR_I2C_FAIL;
    }

    // Verify connection to the chip
    uint8_t test_data;
    if (_i2c_read_byte(vtf, REG_CONFIG_PORT_0, &test_data) != VTF_OK) {
        ESP_LOGE(TAG, "Failed to connect to I2C device at address 0x%02X", i2c_address);
        i2c_driver_delete(i2c_port);
        return VTF_ERR_I2C_FAIL;
    }

    // Set all pins to OUTPUT, except for P00 and P10, which are INPUTs for hold LEDs (00000001)
    _i2c_write_byte(vtf, REG_CONFIG_PORT_0, 0x01);
    _i2c_write_byte(vtf, REG_CONFIG_PORT_1, 0x01);

    // Set all output pins to LOW (00000000)
    _i2c_write_byte(vtf, REG_OUTPUT_PORT_0, 0x00);
    _i2c_write_byte(vtf, REG_OUTPUT_PORT_1, 0x00);

    ESP_LOGI(TAG, "VanTurtle Fan controller initialized at address 0x%02X", i2c_address);

    return VTF_OK;
}

int vanturtlefan_deinit(vanturtlefan_t *vtf) {
    if (vtf == NULL) {
        return VTF_ERR_I2C_FAIL;
    }

    esp_err_t ret = i2c_driver_delete(vtf->i2c_port);
    if (ret != ESP_OK) {
        ESP_LOGE(TAG, "I2C driver delete failed: %s", esp_err_to_name(ret));
        return VTF_ERR_I2C_FAIL;
    }

    return VTF_OK;
}


int vanturtlefan_press(vanturtlefan_t *vtf, uint8_t fan, uint8_t pin, uint16_t length) {
    if (vtf == NULL) {
        return VTF_ERR_I2C_FAIL;
    }

    if (fan != FAN_ONE && fan != FAN_TWO) {
        return VTF_ERR_INVALID_FAN;
    }

    // Use default length if not specified
    if (length == 0) {
        length = BUTTON_PRESS_DURATION;
    }

    // Set pin HIGH to simulate button press
    int ret = _set_pin(vtf, fan, pin, true);
    if (ret != VTF_OK) {
        return ret;
    }

    // Wait for specified duration
    vTaskDelay(pdMS_TO_TICKS(length));

    // Release button by setting pin LOW and turn off LED
    _set_pin(vtf, fan, pin, false);

    return VTF_OK;
}

int vanturtlefan_is_autohold(vanturtlefan_t *vtf, uint8_t fan, bool *state) {
    if (vtf == NULL || state == NULL) {
        return VTF_ERR_I2C_FAIL;
    }

    if (fan != FAN_ONE && fan != FAN_TWO) {
        return VTF_ERR_INVALID_FAN;
    }

    // Get INPUT registers, not OUTPUT
    uint8_t reg = (fan == FAN_ONE) ? REG_INPUT_PORT_0 : REG_INPUT_PORT_1;

    // Read all input states
    uint8_t input_state;
    int ret = _i2c_read_byte(vtf, reg, &input_state);
    if (ret != VTF_OK) {
        return ret;
    }

    // Get P00 or P10 from least significant bit
    // Because the fan pulls low when LED is on, we need to invert the result
    *state = !(input_state & 0x01);

    return VTF_OK;
}

// ============================================================================
// RJ45 Model Functions
// ============================================================================

int vanturtlefan_rj45_auto(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 2, 0);
}

int vanturtlefan_rj45_reverse(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 3, 0);
}

int vanturtlefan_rj45_faster(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 5, 0);
}

int vanturtlefan_rj45_slower(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 6, 0);
}

int vanturtlefan_rj45_beep(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 7, 0);
}

int vanturtlefan_rj45_onoff(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 4, BUTTON_PRESS_DURATION * 2);
}

// ============================================================================
// RJ11 Model Functions
// ============================================================================

int vanturtlefan_rj11_on(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 3, 0);
}

int vanturtlefan_rj11_close(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 4, 0);
}

int vanturtlefan_rj11_open(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 5, 0);
}

int vanturtlefan_rj11_off(vanturtlefan_t *vtf, uint8_t fan) {
    return vanturtlefan_press(vtf, fan, 6, 0);
}
