import smbus
import time

class VanTurtle_Fan:
	FAN_ONE = 1
	FAN_TWO = 2

	REG_INPUT_PORT_0 = 0x00
	REG_INPUT_PORT_1 = 0x01

	REG_OUTPUT_PORT_0 = 0x02
	REG_OUTPUT_PORT_1 = 0x03

	REG_CONFIG_PORT_0 = 0x06
	REG_CONFIG_PORT_1 = 0x07

	BUTTON_PRESS_DURATION = 0.15

	_bus_instances = {}

	def __init__(self, bus=1, address=0x27, no_blink=False):
		"""
		Initialize the VanTurtle Fan controller.

		:param bus: The I2C bus number (default is 1 on pi5).
		:param address: The I2C address of the fan controller, modifiable with the A0, A1 and A2 switches (default is 0x27).
		:param no_blink: If True, do not blink the LEDs on startup (default is False).
		"""

		# Connect to the I2C bus (shared singleton per bus number)
		if bus not in VanTurtle_Fan._bus_instances:
			VanTurtle_Fan._bus_instances[bus] = smbus.SMBus(bus)

		self._bus = VanTurtle_Fan._bus_instances[bus]
		self._address = address

		# Verify connection to the chip
		try:
			self._bus.read_byte_data(self._address, self.REG_CONFIG_PORT_0)
		except Exception as e:
			raise ConnectionError(f"Failed to connect to I2C device at address {hex(self._address)}, did you set the correct address?") from e

		# Set all pins to OUTPUT, except for P00 and P10, which are INPUTs for the hold LEDs (00000001)
		self._bus.write_byte_data(self._address, self.REG_CONFIG_PORT_0, 0x01)
		self._bus.write_byte_data(self._address, self.REG_CONFIG_PORT_1, 0x01)

		# Set all output pins to LOW (00000000)
		self._bus.write_byte_data(self._address, self.REG_OUTPUT_PORT_0, 0x00)
		self._bus.write_byte_data(self._address, self.REG_OUTPUT_PORT_1, 0x00)

		# Blink the LEDs indicate the controller is connected
		if not no_blink:
			self.led(self.FAN_ONE, True)
			self.led(self.FAN_TWO, True)
			time.sleep(self.BUTTON_PRESS_DURATION)
			self.led(self.FAN_ONE, False)
			self.led(self.FAN_TWO, False)

	def _getreg(self, fan):
		"""
		Returns the register for the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO
		:return: The register address for the specified fan.
		"""

		if fan not in (self.FAN_ONE, self.FAN_TWO):
			raise ValueError("Invalid fan number. Use FAN_ONE or FAN_TWO.")

		return self.REG_OUTPUT_PORT_0 if fan == 1 else self.REG_OUTPUT_PORT_1

	def _setpin(self, fan, pin, state):
		"""
		Sets the state of a specific pin for the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO
		:param pin: The pin number to set (1-7).
		:param state: True to set the pin HIGH, False to set it LOW.
		:return: Pin number if the operation was successful.
		"""

		reg = self._getreg(fan)
		# Read the current output state
		output = self._bus.read_byte_data(self._address, reg)

		# Set the requested pin to 0 or 1
		if state:
			output |= 1 << pin
		else:
			output &= ~(1 << pin)

		# Ensure activity LED (bit 1) reflects other bits in this register.
		# If any other bit besides bit 1 is set, set bit 1 to 1; otherwise clear it.
		other_bits_mask = ~(1 << 1) & 0xFF
		if output & other_bits_mask:
			output |= 1 << 1
		else:
			output &= ~(1 << 1)

		# Overwrite the state
		self._bus.write_byte_data(self._address, reg, output)

		return pin

	def press(self, fan, pin, length=False):
		"""
		Simulates a button press for the specified fan and pin.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO
		:param pin: The pin number to press (1-7).
		:param length: Optional duration to hold the button down in seconds. If omitted, uses BUTTON_PRESS_DURATION.
		:return: Pin number if the operation was successful.
		"""

		# Default length is the time to hold the button down
		if length == False: length = self.BUTTON_PRESS_DURATION

		# Always turn on the LED to indicate activity
		# Do this first to that if the script crashes, the LED will still be on
		self.led(fan, True)
		# Set the pin to HIGH to simulate a button press
		self._setpin(fan, pin, True)

		# Wait for the specified length of time, the fan only polls once every +/- 100ms
		time.sleep(length)

		# Release the button by setting the pin to LOW and turn off the LED
		self._setpin(fan, pin, False)
		self.led(fan, False)

		return pin

class VanTurtle_RJ45(VanTurtle_Fan):
	def __init__(self, bus=1, address=0x27, fan=None, no_blink=False):
		"""
		Initialize the VanTurtle RJ45 Fan controller.

		:param bus: The I2C bus number (default is 1 on pi5).
		:param address: The I2C address of the fan controller (default is 0x27).
		:param fan: Default fan to control (FAN_ONE or FAN_TWO). If set, you don't need to specify fan in method calls.
		:param no_blink: If True, do not blink the LEDs on startup (default is False).
		"""
		super().__init__(bus, address, no_blink)
		self._default_fan = fan

	def is_autohold(self, fan):
		"""
		Returns the state of the Auto ("hold to set") LED for the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: True if the Auto LED is on, False if it is off.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan not in (self.FAN_ONE, self.FAN_TWO):
			raise ValueError("Invalid fan number. Use FAN_ONE or FAN_TWO.")

		# Get INPUT registers, not OUTPUT
		reg = self.REG_INPUT_PORT_0 if fan == 1 else self.REG_INPUT_PORT_1

		# Read all input states
		input_state = self._bus.read_byte_data(self._address, reg)

		# Get P00 or P10 from the least significant bit
		# Because the fan pulls low when the LED is on, we need to invert the result
		return not bool(input_state & 0x01)

	# Shorthand functions for specific button presses
	def auto(self, fan=None):
		"""
		Turn Auto on or off on the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 2)

	def reverse(self, fan=None):
		"""
		Reverse airflow direction for the specified fan, button is labeled "In/Out".

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 3)

	def faster(self, fan=None):
		"""
		Increase airflow speed for the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 5)

	def slower(self, fan=None):
		"""
		Lower airflow speed for the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 6)

	def beep(self, fan=None):
		"""
		Makes the specified fan beep with no change to its configuration.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 7)


	# On off has some special handling
	def onoff(self, fan=None):
		"""
		Turn on or off the specified fan, depending on its previous state.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 4, self.BUTTON_PRESS_DURATION * 2)

	# Turning on auto will make the fan turn on when off, and stay on when already on
	# Therefore, if we turn on auto and then turn the fan off we know for sure it is off
	def reset(self, fan=None):
		"""
		Ensures the fan is off by pressing the Auto button followed by the On/Off button.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		self.auto(fan)
		time.sleep(self.BUTTON_PRESS_DURATION)
		return self.onoff(fan)

class VanTurtle_RJ11(VanTurtle_Fan):
	def __init__(self, bus=1, address=0x27, fan=None, no_blink=False):
		"""
		Initialize the VanTurtle RJ11 Fan controller.

		:param bus: The I2C bus number (default is 1 on pi5).
		:param address: The I2C address of the fan controller (default is 0x27).
		:param fan: Default fan to control (FAN_ONE or FAN_TWO). If set, you don't need to specify fan in method calls.
		:param no_blink: If True, do not blink the LEDs on startup (default is False).
		"""
		super().__init__(bus, address, no_blink)
		self._default_fan = fan

	def on(self, fan=None):
		"""
		Turn the fan on if it is off, or cycle through the 4 speed settings if it is already on.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 3)

	def close(self, fan=None):
		"""
		Close the lid of the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 4)

	def open(self, fan=None):
		"""
		Open the lid of the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 5)

	def off(self, fan=None):
		"""
		Turn off the specified fan.

		:param fan: 1 for FAN_ONE, 2 for FAN_TWO. If not specified, uses the default fan set in constructor.
		:return: Pin number if the operation was successful.
		"""
		fan = fan if fan is not None else self._default_fan
		if fan is None:
			raise ValueError("No fan specified. Either pass fan parameter or set default fan in constructor.")
		return self.press(fan, 6)
